# ui/toolbar_widget.py

import os
from PyQt5.QtWidgets import (
    QWidget, QHBoxLayout, QCheckBox, QComboBox, QPushButton, QToolButton
)
from PyQt5.QtGui import QIcon, QDesktopServices
from PyQt5.QtCore import Qt, pyqtSignal, QUrl, QSize
from config import set_locale, t
from utils.path_utils import resource_path


class TopBarWidget(QWidget):
    settings_clicked = pyqtSignal()
    theme_changed = pyqtSignal(bool)
    language_changed = pyqtSignal(str)
    platform_changed = pyqtSignal(str)  # adobe / shutterstock

    def __init__(self, parent=None, dark_mode=False):
        super().__init__(parent)

        layout = QHBoxLayout(self)
        layout.addStretch()

        # Тёмная тема
        self.dark_mode = QCheckBox(t("label_dark_mode"))
        self.dark_mode.setChecked(dark_mode)
        self.dark_mode.stateChanged.connect(self._on_theme_toggle)
        layout.addWidget(self.dark_mode)

        # Язык
        self.lang_combo = QComboBox()
        self.lang_combo.addItem("English", "en")
        self.lang_combo.addItem("Español", "es")
        self.lang_combo.addItem("中文", "zh")
        self.lang_combo.addItem("Deutsch", "de")
        self.lang_combo.addItem("Français", "fr")
        self.lang_combo.addItem("Українська", "uk")
        self.lang_combo.addItem("Русский", "ru")
        self.lang_combo.addItem("Қазақша", "kk")

        locale = os.getenv("APP_LOCALE", "en")
        idx = self.lang_combo.findData(locale)
        if idx >= 0:
            self.lang_combo.setCurrentIndex(idx)
        self.lang_combo.currentIndexChanged.connect(self._on_language_change)
        layout.addWidget(self.lang_combo)

        # === Платформы ===
        self.btn_adobe = QToolButton()
        self.btn_adobe.setIcon(QIcon(resource_path("assets/icons/adobe-stock.png")))
        self.btn_adobe.setCheckable(True)
        self.btn_adobe.setChecked(True)
        self.btn_adobe.setToolTip("Adobe Stock")
        self.btn_adobe.clicked.connect(lambda: self.set_platform("adobe"))
        layout.addWidget(self.btn_adobe)

        self.btn_shutter = QToolButton()
        self.btn_shutter.setIcon(QIcon(resource_path("assets/icons/shutterstock.png")))
        self.btn_shutter.setCheckable(True)
        self.btn_shutter.setChecked(False)
        self.btn_shutter.setToolTip("Shutterstock")
        self.btn_shutter.clicked.connect(lambda: self.set_platform("shutterstock"))
        layout.addWidget(self.btn_shutter)

        # Кнопка глобуса (сайт UMION)
        self.btn_site = QToolButton()
        self.btn_site.setIcon(QIcon(resource_path("assets/icons/globe icon.png")))
        self.btn_site.setToolTip(t("tooltip_umion_site"))
        self.btn_site.setIconSize(QSize(18, 18))
        self.btn_site.setFixedSize(28, 28)
        self.btn_site.clicked.connect(lambda: QDesktopServices.openUrl(QUrl("https://www.umion.xyz/stockmeta.html")))
        layout.addWidget(self.btn_site)

        # Кнопка настроек
        self.btn_settings = QPushButton()
        self.btn_settings.setIcon(QIcon(resource_path("assets/icons/settings.png")))
        self.btn_settings.setToolTip(t("tooltip_api_settings"))
        self.btn_settings.setFixedSize(28, 28)
        self.btn_settings.clicked.connect(self.settings_clicked.emit)
        layout.addWidget(self.btn_settings)

        self.setLayout(layout)

    def _on_theme_toggle(self, state: int):
        self.theme_changed.emit(state == Qt.Checked)

    def _on_language_change(self, index: int):
        locale = self.lang_combo.itemData(index)
        set_locale(locale)
        self.language_changed.emit(locale)

    def update_texts(self):
        self.dark_mode.setText(t("label_dark_mode"))
        self.btn_settings.setToolTip(t("tooltip_api_settings"))
        self.btn_site.setToolTip(t("tooltip_umion_site"))

    def set_platform(self, name: str):
        if name == "adobe":
            self.btn_adobe.setChecked(True)
            self.btn_shutter.setChecked(False)
        else:
            self.btn_adobe.setChecked(False)
            self.btn_shutter.setChecked(True)
        self.platform_changed.emit(name)

    def current_platform(self) -> str:
        return "adobe" if self.btn_adobe.isChecked() else "shutterstock"

    def set_platform(self, name: str):
        if name == "adobe":
            self.btn_adobe.setChecked(True)
            self.btn_shutter.setChecked(False)
        else:
            self.btn_adobe.setChecked(False)
            self.btn_shutter.setChecked(True)

        self.btn_adobe.setStyleSheet("""
            QToolButton {
                border: 2px solid #00BFFF;
                border-radius: 6px;
                background-color: rgba(255,255,255,40);
            }
        """ if name == "adobe" else "QToolButton { background: none; }")

        self.btn_shutter.setStyleSheet("""
            QToolButton {
                border: 2px solid #FF4500;
                border-radius: 6px;
                background-color: rgba(255,255,255,40);
            }
        """ if name == "shutterstock" else "QToolButton { background: none; }")
        self.platform_changed.emit(name)


